// Apple ][ Audio Uploader
// Copyright (c) Sebastian Kienzl <seb at riot dot org>
// 
// $Id: main.cpp 30 2008-09-16 10:29:40Z seb $
//
// Information on the Waveform taken from "Woz Wonderbook",
// ADTPro-Source and Monitor-ROM disassembly
//
// Should also compile under Linux and other POSIX-platforms;
// of course with the playback-functionality disabled

#include <string>
#include <iostream>
#include <sstream>
#include <stdio.h>
#include <stdlib.h>
#include "wav.h"
#ifdef _WIN32
#include "getopt_win.h"
#include "waveout.h"
#endif

using namespace std;

int longCycle, shortCycle;
int sampleFrequency = 44100;
int pilotMS = 3000;
int amplitude = 100;

class ByteBuffer {
public:
	ByteBuffer() : data(0), allocLen(0), len(0) {}
	~ByteBuffer() { if( data ) free( data ); }

	void addByte( char d ) {
		if( len >= allocLen ) {
			#define ALLOC_STEP 1024
			data = (char*)realloc( data, allocLen + ALLOC_STEP );
			memset( data + allocLen, 0, ALLOC_STEP );
			allocLen += ALLOC_STEP;
		}
		data[ len++ ] = d;
	}

	int getLength() { return len; }
	const char* getData() { return data; }
private:
	char* data;
	int allocLen;
	int len;
};

void addCycle( ByteBuffer& b, int hz ) {
	int len = sampleFrequency / hz;

	for( int i = 0; i < len / 2; i++ ) {
		b.addByte( amplitude + 0x80 );
	}
	for( int i = len / 2; i < len; i++ ) {
		b.addByte( -amplitude + 0x80 );
	}
}

void encodeByte( ByteBuffer& b, s8 byte ) {
	for( int i = 0; i < 8; i++ ) {
		addCycle( b, byte & (1<<7) ? 1000 : 2000 );
		byte <<= 1;
	}
}

void usage( const char* argv0 ) {
	cerr << "Save as WAV: " << endl << "\t" << argv0 << " -i <infile> -o <outfile> [-f samplefreq] [-p leadingtone_msec]" << endl;
#ifdef _WIN32	
	cerr << "Enumerate sound devices: " << endl << "\t" << argv0 << " -l" << endl;
	cerr << "Output sound: " << endl << "\t" << argv0 << " -i <infile> [-f samplefreq] [-p leadingtone_msec] [-d device]" << endl;
#endif
	cerr << endl;
	exit( 1 );
}

int main( int argc, char** argv )
{
	cout << "Apple ][ Audio Uploader" << endl;
	cout << "(c) 2008 seb at riot dot org" << endl;
	cout << "$Rev: 30 $" << endl << endl;
	int opt;
	string inFile, outFile;
	int sampleFrequency = 44100;
	int size;

	bool saveFile = false;
#ifdef _WIN32
	int playbackDevice = -1;
#endif

	while( ( opt = getopt( argc, argv,
#ifdef _WIN32 
		"f:i:o:d:p:l"
#else
		"f:i:o:p:"
#endif
	) ) >= 0 ) {
		switch( opt ) {
		case 'f':
			sampleFrequency = atoi( optarg );
			break;
		case 'i':
			inFile = optarg;
			break;
		case 'o':
			outFile = optarg;
			saveFile = true;
			break;
		case 'p':
			pilotMS = atoi( optarg );
			break;
#ifdef _WIN32
		case 'd':
			playbackDevice = atoi( optarg );
			break;
		case 'l':
			enumerateWaveout();
			return 0;
#endif
		case 'a':
			amplitude = atoi( optarg );
			break;
		default:
			usage( argv[0] );
			break;
		}
	}

	if( inFile.empty()
#ifndef _WIN32
		|| outFile.empty()
#endif
	)
		usage( argv[0] );

	FILE* fin = fopen( inFile.c_str(), "rb" );
	if( !fin ) {
		cerr << "Can't open file " << inFile << "!" << endl;
		return 1;
	}

	fseek( fin, 0, SEEK_END );
	size = ftell( fin );
	fseek( fin, 0, SEEK_SET );

	u8* data = new u8[ size ];
	if( fread( data, size, 1, fin ) != 1 ) {
		cerr << "Can't read file " << inFile << "!" << endl;
		return 1;
	}
	fclose( fin );

	cout << "File " << inFile << ": " << size << " bytes" << endl;

	ByteBuffer buff;

	longCycle = (int)( 0.001f * sampleFrequency + .5f );
	shortCycle = (int)( 0.0005f * sampleFrequency + .5f );

	// 2*650usec => 769 Hz
	for( int i = 0; i < pilotMS * 1000 / 769; i++ ) {
		addCycle( buff, 769 );
	}
	// start-cycle
	addCycle( buff, 2200 );

	u8 checksum = 0xff;
	for( int i = 0; i < size; i++ ) {
		u8 byte = data[ i ];
		checksum ^= byte;
		encodeByte( buff, byte );
	}
	encodeByte( buff, checksum );
	addCycle( buff, 200 );

	delete [] data;
#ifdef _WIN32
	if( !saveFile ) {
		playWave( sampleFrequency, (const u8*)buff.getData(), buff.getLength(), playbackDevice );
	}
	else {
#endif
		if( !saveFile8Bit( outFile, (const u8*)buff.getData(), buff.getLength(), sampleFrequency ) ) {
			return 1;
		}

		cout << "File " << outFile << " written" << endl;
#ifdef _WIN32
	}
#endif
	
	return 0;
}
